<?php
// app/Services/AIPredictionService.php

namespace App\Services;

use App\Models\Desa;
use App\Models\Prediksi;
use App\Models\Laporan;
use Carbon\Carbon;

class AIPredictionService
{
    protected $bmkgService;
    
    public function __construct(BMKGService $bmkgService)
    {
        $this->bmkgService = $bmkgService;
    }
    
    /**
     * Prediksi risiko kekeringan untuk 7 hari ke depan
     */
    public function predictDroughtRisk($desaId, $days = 7)
    {
        $desa = Desa::find($desaId);
        if (!$desa) return [];
        
        $predictions = [];
        
        for ($i = 0; $i < $days; $i++) {
            $targetDate = Carbon::now()->addDays($i);
            $cuacaData = $this->bmkgService->getCurahHujanDesa($desaId, $targetDate->format('Y-m-d'));
            
            // AI Algorithm: Weighted scoring system
            $risikoScore = $this->calculateRiskScore(
                $cuacaData['curah_hujan'],
                $cuacaData['kelembapan'],
                $desa,
                $targetDate
            );
            
            $tingkatRisiko = $this->determineTingkatRisiko($risikoScore);
            $rekomendasi = $this->generateRekomendasi($tingkatRisiko, $risikoScore);
            
            // Simpan ke database
            $prediksi = Prediksi::updateOrCreate(
                [
                    'desa_id' => $desaId,
                    'tanggal_prediksi' => $targetDate->format('Y-m-d'),
                ],
                [
                    'curah_hujan' => $cuacaData['curah_hujan'],
                    'kelembapan_tanah' => $cuacaData['kelembapan'],
                    'tingkat_risiko' => $tingkatRisiko,
                    'risiko_score' => $risikoScore,
                    'rekomendasi' => $rekomendasi,
                    'data_bmkg' => json_encode($cuacaData),
                ]
            );
            
            $predictions[] = $prediksi;
        }
        
        return $predictions;
    }
    
    /**
     * Calculate risk score menggunakan weighted algorithm
     */
    protected function calculateRiskScore($curahHujan, $kelembapan, $desa, $targetDate)
    {
        $score = 0;
        
        // Factor 1: Curah Hujan (40% weight)
        if ($curahHujan < 10) {
            $score += 40;
        } elseif ($curahHujan < 50) {
            $score += 25;
        } elseif ($curahHujan < 100) {
            $score += 10;
        }
        
        // Factor 2: Kelembapan Tanah (30% weight)
        if ($kelembapan < 30) {
            $score += 30;
        } elseif ($kelembapan < 50) {
            $score += 20;
        } elseif ($kelembapan < 70) {
            $score += 10;
        }
        
        // Factor 3: Historical Data - Laporan warga (20% weight)
        $recentReports = Laporan::where('desa_id', $desa->id)
            ->where('created_at', '>=', now()->subDays(7))
            ->whereIn('jenis_laporan', ['sumur_kering', 'mata_air_kering'])
            ->count();
        
        if ($recentReports > 5) {
            $score += 20;
        } elseif ($recentReports > 2) {
            $score += 10;
        }
        
        // Factor 4: Musim (10% weight)
        $month = $targetDate->month;
        if (in_array($month, [7, 8, 9, 10])) { // Musim kemarau
            $score += 10;
        }
        
        return min($score, 100);
    }
    
    /**
     * Determine tingkat risiko berdasarkan score
     */
    protected function determineTingkatRisiko($score)
    {
        if ($score >= 70) {
            return 'kritis';
        } elseif ($score >= 40) {
            return 'waspada';
        }
        return 'aman';
    }
    
    /**
     * Generate rekomendasi berdasarkan tingkat risiko
     */
    protected function generateRekomendasi($tingkatRisiko, $score)
    {
        return match($tingkatRisiko) {
            'kritis' => 'PERINGATAN KRITIS! Segera lakukan penghematan air maksimal. Batasi penggunaan air untuk kebutuhan vital saja. Koordinasi dengan pemdes untuk distribusi air bersih.',
            'waspada' => 'Tingkat risiko sedang. Mulai hemat penggunaan air. Hindari pemborosan air untuk mencuci kendaraan atau menyiram tanaman non-esensial.',
            'aman' => 'Kondisi air masih aman. Tetap gunakan air secara bijak dan pantau perkembangan cuaca.',
            default => 'Monitoring kondisi air terus dilakukan.',
        };
    }
    
    /**
     * AI untuk scheduling tangki air (Optimization Algorithm)
     */
    public function optimizeTangkiSchedule()
    {
        // Ambil semua desa dengan risiko tinggi
        $kriticalDesa = Prediksi::with('desa')
            ->where('tanggal_prediksi', '>=', now()->format('Y-m-d'))
            ->where('tingkat_risiko', 'kritis')
            ->orderBy('risiko_score', 'desc')
            ->get()
            ->groupBy('desa_id');
        
        $schedule = [];
        $prioritas = 10;
        
        foreach ($kriticalDesa as $desaId => $prediksiList) {
            $avgScore = $prediksiList->avg('risiko_score');
            $desa = $prediksiList->first()->desa;
            
            $schedule[] = [
                'desa_id' => $desaId,
                'desa_nama' => $desa->nama_desa,
                'prioritas' => $prioritas--,
                'risiko_score' => round($avgScore, 2),
                'volume_recommended' => $this->calculateWaterNeed($desa),
                'tanggal_rekomendasi' => now()->addDay()->format('Y-m-d'),
            ];
        }
        
        return $schedule;
    }
    
    /**
     * Hitung kebutuhan air
     */
    protected function calculateWaterNeed($desa)
    {
        // Asumsi: 50 liter per orang per hari untuk 3 hari
        return $desa->jumlah_penduduk * 50 * 3;
    }
}
